#include "driver.h"
#include "gp32_menu.h"

#define WAITKEY		while(!osd_key_pressed(OSD_KEY_PAUSE)) { Delay(100); osd_poll_joystick();}Delay(200);osd_poll_joystick();
#define MEM		{char sMsg[64];gm_sprintf(sMsg, "Avail: %d", gm_availablesize());gp32_text_log(sMsg);}
#define DEBUG 0

char mameversion[] = "0.34 ("__DATE__")";

static struct RunningMachine machine;
struct RunningMachine *Machine = &machine;
static const struct GameDriver *gamedrv;
static const struct MachineDriver *drv;

int nocheat;    /* 0 when the -cheat option was specified */

int frameskip;
int VolumePTR = 0;
static int settingsloaded;

unsigned char *ROM;

int init_machine(void);
void shutdown_machine(void);
int run_machine(void);


int run_game(int game, struct GameOptions *options)
{
	int err;

	Machine->gamedrv = gamedrv = drivers[game];
	Machine->drv = drv = gamedrv->drv;

	/* copy configuration */
	Machine->sample_rate = options->samplerate;
	Machine->sample_bits = options->samplebits;
	frameskip = options->frameskip;
	nocheat = 1;

	/* get orientation right */
	Machine->orientation = gamedrv->orientation;
	if (options->norotate)
		Machine->orientation = ORIENTATION_DEFAULT;
	if (options->ror)
	{
		/* if only one of the components is inverted, switch them */
		if ((Machine->orientation & ORIENTATION_ROTATE_180) == ORIENTATION_FLIP_X ||
				(Machine->orientation & ORIENTATION_ROTATE_180) == ORIENTATION_FLIP_Y)
			Machine->orientation ^= ORIENTATION_ROTATE_180;

		Machine->orientation ^= ORIENTATION_ROTATE_90;
	}
	if (options->rol)
	{
		/* if only one of the components is inverted, switch them */
		if ((Machine->orientation & ORIENTATION_ROTATE_180) == ORIENTATION_FLIP_X ||
				(Machine->orientation & ORIENTATION_ROTATE_180) == ORIENTATION_FLIP_Y)
			Machine->orientation ^= ORIENTATION_ROTATE_180;

		Machine->orientation ^= ORIENTATION_ROTATE_270;
	}
	if (options->flipx)
		Machine->orientation ^= ORIENTATION_FLIP_X;
	if (options->flipy)
		Machine->orientation ^= ORIENTATION_FLIP_Y;


	/* Do the work*/
	err = 1;

	if (init_machine() == 0)
	{
		if (osd_init() == 0)
		{
			if (run_machine() == 0)
				err = 0;
			else gp32_text_log("Unable to start machine emulation\n");

#if DEBUG
			gp32_text_log("osd_exit()\n");
			WAITKEY
#endif
			osd_exit();
		}
		else gp32_text_log("Unable to initialize system\n");

#if DEBUG
		gp32_text_log("shutdown_machine()\n");
		WAITKEY
#endif
		shutdown_machine();
	}
	else gp32_text_log("Unable to initialize machine emulation\n");

	return err;
}



/***************************************************************************

  Initialize the emulated machine (load the roms, initialize the various
  subsystems...). Returns 0 if successful.

***************************************************************************/
int init_machine(void)
{
	if (gamedrv->input_ports)
	{
		int total;
		const struct InputPort *from;
		struct InputPort *to;

		from = gamedrv->input_ports;

		total = 0;
		do
		{
			total++;
		} while ((from++)->type != IPT_END);

		if ((Machine->input_ports = gm_zi_malloc(total * sizeof(struct InputPort))) == 0)
			return 1;

		from = gamedrv->input_ports;
		to = Machine->input_ports;

		do
		{
			asm_memcpy(to,from,sizeof(struct InputPort));

			to++;
		} while ((from++)->type != IPT_END);
	}

	if (readroms() != 0)
	{
		gp32_text_log("Error reading ROMs");
		gm_free(Machine->input_ports);
		return 1;
	}

	{
		extern unsigned char *RAM;
		RAM = Machine->memory_region[drv->cpu[0].memory_region];
		ROM = RAM;
	}

	/* decrypt the ROMs if necessary */
	if (gamedrv->rom_decode) {
		gp32_text_log("Decoding ROMs...");
		(*gamedrv->rom_decode)();
	}

	if (gamedrv->opcode_decode)
	{
		int j;
		extern int encrypted_cpu;

		gp32_text_log("Decoding Opcodes...");

		/* find the first available memory region pointer */
		j = 0;
		while (Machine->memory_region[j]) j++;

		/* allocate a ROM array of the same length of memory region #0 */
		if ((ROM = gm_zi_malloc(Machine->memory_region_length[0])) == 0)
		{
			gm_free(Machine->input_ports);
			/* TODO: should also free the allocated memory regions */
			return 1;
		}

		Machine->memory_region[j] = ROM;
		Machine->memory_region_length[j] = Machine->memory_region_length[0];

		encrypted_cpu = 0;
		(*gamedrv->opcode_decode)();
	}


	/* read audio samples if available */
	gp32_text_log("Loading Samples...");
	Machine->samples = readsamples(gamedrv->samplenames,gamedrv->name);


	/* first of all initialize the memory handlers, which could be used by the */
	/* other initialization routines */
	gp32_text_log("Initializing CPU...");
	cpu_init();

	/* load input ports settings (keys, dip switches, and so on) */
	gp32_text_log("Loading Input Ports Settings...");
	settingsloaded = load_input_port_settings();

	/* ASG 971007 move from mame.c */
	gp32_text_log("Initializing Memory Handlers...");
	if( !initmemoryhandlers() )
	{
		gm_free(Machine->input_ports);
		return 1;
	}


	gp32_text_log("Driver Init...");
	if (gamedrv->driver_init) (*gamedrv->driver_init)();

	return 0;
}



void shutdown_machine(void)
{
	int i;

	/* free audio samples */
	freesamples(Machine->samples);
	Machine->samples = 0;

	/* ASG 971007 free memory element map */
	shutdownmemoryhandler();

	/* free the memory allocated for ROM and RAM */
	for (i = 0;i < MAX_MEMORY_REGIONS;i++)
	{
		gm_free(Machine->memory_region[i]);
		Machine->memory_region[i] = 0;
		Machine->memory_region_length[i] = 0;
	}

	/* free the memory allocated for input ports definition */
	gm_free(Machine->input_ports);
	Machine->input_ports = 0;
}



static void vh_close(void)
{
	int i;


	for (i = 0;i < MAX_GFX_ELEMENTS;i++)
	{
		freegfx(Machine->gfx[i]);
		Machine->gfx[i] = 0;
	}
	osd_close_display();
	palette_stop();
}



static int vh_open(void)
{
	int i;
	int visible_width;
	int visible_height;


	for (i = 0;i < MAX_GFX_ELEMENTS;i++) Machine->gfx[i] = 0;
	Machine->uifont = 0;

	if (palette_start() != 0)
	{
		vh_close();
		return 1;
	}

	/* convert the gfx ROMs into character sets. This is done BEFORE calling the driver's */
	/* convert_color_prom() routine (in palette_init()) because it might need to check the */
	/* Machine->gfx[] data */
	if (drv->gfxdecodeinfo)
	{
		for (i = 0;i < MAX_GFX_ELEMENTS && drv->gfxdecodeinfo[i].memory_region != -1;i++)
		{
			if ((Machine->gfx[i] = decodegfx(Machine->memory_region[drv->gfxdecodeinfo[i].memory_region] + drv->gfxdecodeinfo[i].start,
					drv->gfxdecodeinfo[i].gfxlayout)) == 0)
			{
				vh_close();
				gp32_text_log("Out of Memory: decodegfx()");
				return 1;
			}
			Machine->gfx[i]->colortable = &Machine->colortable[drv->gfxdecodeinfo[i].color_codes_start];
			Machine->gfx[i]->total_colors = drv->gfxdecodeinfo[i].total_color_codes;
		}
	}

#ifdef EARLY_GFX_DISPOSE
			/* Free DISPOSABLE ROMs now
			 * Can't wait to dispose of decoded gfx RAM after vh_open and vh_start have finished
			 * This is a bit of a hack but is necessary for the GP32 with it's 8MB of RAM.
			 * There are no checks here and two assumptions:
			 * Disposable regions start from region 1
			 * Number of sections in gfxdecode match the number of memory regions to dispose
			 * For most drivers this is NOT the case and must be modified accordingly.
			 */
/*
			{
				char sMsg[40];
				gm_sprintf(sMsg, "Freeing memory region [%d]", i+1);
				gp32_text_log(sMsg);
				WAITKEY
			}
			gm_free(Machine->memory_region[i+1]);               
			Machine->memory_region[i+1] = 0;                             
*/
			/*
			 * Hack for Roadblasters, free on every second call
			if (!strcmp(Machine->gamedrv->name,"roadblst") && !(i%2))
			{
				continue;
			}
			 */

			{
                                const struct RomModule *romp = gamedrv->rom;
                                int     region;

                                /* free memory regions allocated with ROM_REGION_DISPOSE (typically gfx roms) */
                                for (region = 0; romp->name || romp->offset || romp->length; region++)
                                {
                                        if (romp->offset & ROMFLAG_DISPOSE)
                                        {
                                                if (Machine->memory_region[region])
                                                {
#if 0
                                                        {
                                                                char sMsg[40];
                                                                gm_sprintf(sMsg, "Freeing memory region [%d]", region);
                                                                gp32_text_log(sMsg);
                                                        }
#endif
                                                        gm_free (Machine->memory_region[region]);
                                                        Machine->memory_region[region] = 0;
                                                }
                                        }
                                        do { romp++; } while (romp->length);
                                }
			}
#endif
	
	/* create the display bitmap, and allocate the palette */
	/* ppf 16-9-05 */
	visible_width=(drv->visible_area.max_x-drv->visible_area.min_x)+1;
	visible_height=(drv->visible_area.max_y-drv->visible_area.min_y)+1;
	if ((Machine->scrbitmap = osd_create_display(
			drv->screen_width,drv->screen_height,
			drv->video_attributes,
			visible_width,visible_height, /* Visible zone size */
			drv->visible_area.min_x, drv->visible_area.min_y)) == 0) /* Visible zone offset */
	{
		vh_close();
		return 1;
	}
	/* fin ppf 16-9-05 */

	/* initialize the palette - must be done after osd_create_display() */
	palette_init();

	return 0;
}


/***************************************************************************

  This function takes care of refreshing the screen, processing user input,
  and throttling the emulation speed to obtain the required frames per second.

***************************************************************************/

int updatescreen(void)
{
	static int framecount = 0;
	int skipme = 1;
	int i;


	/* see if we recomend skipping this frame */
	if (++framecount > frameskip)
	{
		framecount = 0;
		skipme = 0;
	}

	/* if not, go for it */
	if (!skipme)
		(*drv->vh_update)(Machine->scrbitmap,0);  /* update screen */

	/* if the user pressed F3, reset the emulation */
	if (osd_key_pressed(OSD_KEY_RESET_MACHINE)) {
		while(osd_key_pressed(OSD_KEY_RESET_MACHINE)) { osd_poll_joystick(); }
		machine_reset();
	}

	if (osd_key_pressed(OSD_KEY_PAUSE)) /* pause the game */
	{
		for(i=0;i<10;i++) {
			if (osd_key_pressed(OSD_KEY_CANCEL)) return 1;
			if (!osd_key_pressed(OSD_KEY_PAUSE)) break;
			osd_poll_joystick();
			Delay(100);
		}
		
		/* Screen change orientation */
		if(i<10) {
			extern int gp32_new_gfx_core;
			gp32_rotate=(gp32_rotate==0?1:0);
			if (gp32_new_gfx_core)
				gp32_adjust_display();
			else
				gp32_adjust_display_old();
			gp32_clear_screen();
			if (!skipme)
				osd_update_display();
			return 0;
		}

		osd_sound_enable(0);
		
		gp32_text_pause();
		while(osd_key_pressed(OSD_KEY_PAUSE)) {
			Delay(100);
			osd_poll_joystick();
		}
		
		/* Wait Unpause */
		while(!osd_key_pressed(OSD_KEY_PAUSE)) {
			if (osd_key_pressed(OSD_KEY_CANCEL)) return 1;
			Delay(100);
			osd_poll_joystick();
		}
		while(osd_key_pressed(OSD_KEY_PAUSE)) {
			Delay(100);
			osd_poll_joystick();
		}
			
		osd_sound_enable(1);
	}

	if (!skipme)
		osd_update_display();
	
	return 0;
}


/***************************************************************************

  Run the emulation. Start the various subsystems and the CPU emulation.
  Returns non zero in case of error.

***************************************************************************/
int run_machine(void)
{
	int res = 1;

	gp32_text_log("Running Machine...");


	gp32_text_log("Decoding Graphics...");
	if (vh_open() == 0)
	{
		gp32_text_log("Video Hardware...");
		if (drv->vh_start == 0 || (*drv->vh_start)() == 0)      /* start the video hardware */
		{
			gp32_text_log("Audio Hardware...");
		
			if (sound_start() == 0) /* start the audio hardware */
			{
#ifndef EARLY_GFX_DISPOSE
				const struct RomModule *romp = gamedrv->rom;
				int	region;

				/* free memory regions allocated with ROM_REGION_DISPOSE (typically gfx roms) */
				for (region = 0; romp->name || romp->offset || romp->length; region++)
				{
					if (romp->offset & ROMFLAG_DISPOSE)
					{
						if (Machine->memory_region[region])
						{
							{
								char sMsg[40];
								gm_sprintf(sMsg, "Freeing memory region [%d]", region);
								gp32_text_log(sMsg);
							}
							gm_free (Machine->memory_region[region]);
							Machine->memory_region[region] = 0;
						}
					}
					do { romp++; } while (romp->length);
				}
#endif
				gp32_clear_screen(); /* Clear GP32 Screen */
				cpu_run();      /* run the emulation! */

				/* the following MUST be done after hiscore_save() otherwise */
				/* some 68000 games will not work */
#if DEBUG
				gp32_text_log("sound_stop()");
				WAITKEY
#endif
				sound_stop();
				if (drv->vh_stop)
				{
#if DEBUG
					gp32_text_log("vh_stop()");
					WAITKEY
#endif
					(*drv->vh_stop)();
				}

				res = 0;
			}
			else 
			{
				gp32_text_log("Unable to start audio emulation");
				WAITKEY
			}
		}
		else
		{
			gp32_text_log("Unable to start video emulation");
			WAITKEY
		}

#if DEBUG
		gp32_text_log("vh_close()");
		WAITKEY
#endif
		vh_close();
	}
	else
	{
		gp32_text_log("Unable to initialize display");
		WAITKEY
	}

	return res;
}
